﻿' Form to collect information for a banquet reservation
Public Class BanquetReservationForm
    ' The default number to set the NumberOfPeopleTextBox
    Public Const PeopleCountDefault As Integer = 30

    ' The minimum number of people for a banquet
    Public Const MinPeopleCount As Integer = 25

    ' The maximum number of people for a banquet
    Public Const MaxPeopleCount As Integer = 1600

    ' Perform run-time initialization.
    Private Sub BanquetReservationForm_Load(ByVal sender As System.Object, _
                                    ByVal e As System.EventArgs) _
      Handles MyBase.Load
        PeopleCountUpDown.Minimum = MinPeopleCount
        PeopleCountUpDown.Maximum = MaxPeopleCount
        PeopleCountUpDown.Text = PeopleCountDefault.ToString

        ' The earliest possible date is tomorrow
        Dim EarliestAllowableDate As DateTime = Today().AddDays(1)
        DatePicker.MinDate = EarliestAllowableDate
        DatePicker.MaxDate = EarliestAllowableDate.AddYears(5)
        DatePicker.Value = EarliestAllowableDate

        Dim mediator As New BanquetReservationMediator(Me)
    End Sub

    ' Handle a click on the OK button.
    Private Sub OKButton_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles OKBtn.Click
        DialogResult = DialogResult.OK
        Hide()
    End Sub

    ' Handle a click on the cancel button.
    Private Sub CancelBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles CancelBtn.Click
        DialogResult = DialogResult.Cancel
        Hide()
    End Sub

    ' This class is responsible for mediating amoung the
    ' components of the a BanquetReservationForm
    Private Class BanquetReservationMediator
        ' The minimum duration for a banquet reservation is 3 hours
        Private Shared ReadOnly MinDuration As TimeSpan _
            = New TimeSpan(3, 0, 0) ' 3 hours

        ' The foods available with table service for breakfast.
        Private Shared ReadOnly TableBreakfast As String() _
                                = {"Omlettes", "Bacon", _
                                "Toast", "Cold Cerial", _
                                "Oatmeal", "Fruit Bowl", _
                                "Eggs Benedict"}

        ' The foods available with table service after breakfast.
        Private Shared ReadOnly TableFood As String() _
                                = {"Roast Beef", "Egg Rolls", _
                                "Shish Kebob", "Burritos", _
                                "Lasagna", "Ham", _
                                "Veal Marsala", _
                                "Saurbraten", _
                                "Beef Wellington", _
                                "Mesquite Chicken", _
                                "Lamb Shanks", "Prime Rib", _
                                "Stuffed Grouper"}

        ' The foods available with buffet service for breakfast.
        Private Shared ReadOnly BuffetBreakfast As String() _
                                = {"Scrambled Eggs", "Bacon", _
                                "Toast", "Cold Cerial", _
                                "Oatmeal", _
                                "Fruit Assortment"}

        ' The foods available with buffet service after breakfast.
        Private Shared ReadOnly BuffetFood As String() _
                                = {"Roast Beef", "Egg Rolls", _
                                "Shish Kebob", "Burritos", _
                                "Lasagna", "Ham", _
                                "Saurbraten", _
                                "Beef Wellington", _
                                "Mesquite Chicken", _
                                "Prime Rib"}

        ' The object whose components this object mediates for.
        Private MyForm As BanquetReservationForm

        ' Constructor
        Public Sub New(ByVal theForm As BanquetReservationForm)
            MyForm = theForm
            AddHandler MyForm.StartTimePicker.ValueChanged, _
                    AddressOf EventReceived
            AddHandler MyForm.EndTimePicker.ValueChanged, _
                    AddressOf EventReceived
            AddHandler MyForm.BuffetLineRadioBtn.CheckedChanged, _
                    AddressOf EventReceived
            AddHandler MyForm.TableServiceRadioBtn.CheckedChanged, _
                    AddressOf EventReceived
            AddHandler MyForm.MenuListBox.SelectedIndexChanged, _
                    AddressOf EventReceived
        End Sub

        ' Handle events by calling EnforceInvariants
        Public Sub EventReceived(ByVal sender As Object, ByVal e As EventArgs)
            EnforceInvariants()
        End Sub

        ' Enforce all invariant relationships between components
        ' in the form.
        Public Sub EnforceInvariants()
            ' Propagate the selected date to the chosen times.
            Dim datePicker As DateTimePicker = MyForm.DatePicker
            Dim startPicker As DateTimePicker = MyForm.StartTimePicker
            Dim endPicker As DateTimePicker = MyForm.EndTimePicker
            Dim startTime As DateTime = startPicker.Value
            Dim endTime As DateTime = endPicker.Value
            startPicker.Value = MergeDateAndTime(datePicker.Value, startTime)
            endPicker.Value = MergeDateAndTime(datePicker.Value, endTime)

            startPicker.MinDate = BeginningOfDay(datePicker.Value)
            endPicker.MaxDate = EndOfDay(datePicker.Value)
            startPicker.MaxDate = endPicker.Value.Subtract(MinDuration)
            endPicker.MinDate = startPicker.Value.Add(MinDuration)

            ' Translate start time and the selection of buffet or 
            ' table(service) into a food menu.
            If MyForm.BuffetLineRadioBtn.Checked Then
                If startTime.Hour <= 8 Or (startTime.Hour <= 9 And _
                                        endTime.Hour <= 12) Then
                    MyForm.MenuListBox.DataSource = BuffetBreakfast
                Else
                    MyForm.MenuListBox.DataSource = BuffetFood
                End If
                MyForm.MenuListBox.Enabled = True
            ElseIf MyForm.TableServiceRadioBtn.Checked Then
                If startTime.Hour <= 10 Then
                    MyForm.MenuListBox.DataSource = TableBreakfast
                Else
                    MyForm.MenuListBox.DataSource = TableFood
                End If
                MyForm.MenuListBox.Enabled = True
            Else
                MyForm.MenuListBox.DataSource = Nothing
                MyForm.MenuListBox.Enabled = False
            End If

            ' Determine if the duration of the banquet is acceptable.
            Dim duration As TimeSpan
            duration = endPicker.Value.Subtract(startPicker.Value)
            Dim durationOk As Boolean
            durationOk = (duration > MinDuration)

            ' Determine if the OK button should be enabled.
            MyForm.OKBtn.Enabled = durationOk _
                And MyForm.MenuListBox.Enabled _
                And MyForm.MenuListBox.SelectedItems.Count > 0
        End Sub

        ' Combine the year, month and day of one DateTime with
        ' the hour, minute and second of another.
        Private Function MergeDateAndTime(ByVal dateSource As DateTime, _
                                        ByVal timeSource As DateTime) As DateTime
            Return New DateTime(dateSource.Year, dateSource.Month, _
                                dateSource.Day, timeSource.Hour, _
                                timeSource.Minute, timeSource.Second)
        End Function

        ' Return the midnight that begins the given day.
        Private Function BeginningOfDay(ByVal day As DateTime) As DateTime
            Return New DateTime(day.Year, day.Month, day.Day)
        End Function

        ' Return the midnight that ends the given day.
        Private Function EndOfDay(ByVal day As DateTime) As DateTime
            Return New DateTime(day.Year, day.Month, day.Day, 23, 59, 59)
        End Function
    End Class ' BanquetReservationMediator
End Class ' BanquetReservationForm
